"""This file contains the ViTModel class that is used as a benchmark game for the `shapiq` library.

Note to developers:
    This file should not be imported directly as it requires a lot of dependencies to be installed
    (e.g. `torch`, `transformers`, `PIL`).
"""
import numpy as np
import torch
import torch.nn.functional as F
from PIL.Image import Image
from torch import nn
from transformers import ViTFeatureExtractor, ViTForImageClassification

__all__ = ["ViTModel"]


class ViTModel:

    """Sets up the Vision Transformer model from huggingface as a callable function.

    Args:
        n_patches: The number of patches to be used. Either 9 or 16.
        input_image: The image to be explained.
        verbose: Whether to print the original class and its probability. Defaults to True.

    Attributes:
        n_patches: The number of patches used.
        empty_value: The output probability of an empty coalition.
        class_id: The original class id of the input image.
        original_output: The original output probability of the input image.
        original_class_name: The original class name of the input image.

    Raises:
        ValueError: If the number of patches is not 9 or 16.
    """

    def __init__(self, n_patches: int, input_image: Image, verbose: bool = True) -> None:
        # check input
        if n_patches not in [9, 16]:
            raise ValueError(f"The number of patches must be either 9 or 16 and not {n_patches}")

        self.n_patches = n_patches
        self.class_id = None  # will be overwritten after we know the original class

        # setup device for model
        device = torch.device("cuda" if torch.cuda.is_available() else "cpu")

        # load model
        feature_extractor = ViTFeatureExtractor.from_pretrained("google/vit-base-patch32-384")
        model = ViTForImageClassification.from_pretrained("google/vit-base-patch32-384")
        model.to(device)

        # get model parts
        self._embedding_layer = model.vit.embeddings
        self._encoder = model.vit.encoder
        self._classifier = model.classifier

        # setup last normalization layer params manually
        self._norm_weight = NORM_WEIGHT
        self._norm_bias = NORM_BIAS
        self._norm_eps = 1e-12
        self._norm_shape = (768,)

        # set mask token of embedding layer to zeros to use `bool_masked_pos` parameter for masking
        self._embedding_layer.mask_token = nn.Parameter(torch.zeros(1, 1, 768))

        # run input image through model
        self._transformed_image = feature_extractor(images=input_image, return_tensors="pt")

        # get original output
        probit_output = self(np.ones(self.n_patches, dtype=bool))
        self.class_id = int(np.argmax(probit_output))
        self.original_output = float(probit_output[0, self.class_id])
        self.original_class_name = str(model.config.id2label[self.class_id])
        if verbose:
            print(f"Original class: {self.original_class_name} ({self.original_output:.4f})")

        # call the model with no information to get empty prediction
        empty_output = self(np.zeros(self.n_patches, dtype=bool))
        self.empty_value = float(empty_output)

    def __call__(self, coalitions: np.ndarray) -> np.ndarray:
        """Returns the class probability of the coalition.

        Args:
            coalitions: The coalition of players (i.e. super-patches).

        Returns:
            The class probability of the coalition.
        """
        # make coalition into 2d
        if len(coalitions.shape) == 1:
            coalitions = coalitions.reshape((1, -1))

        n_coalitions = coalitions.shape[0]
        probit_output_all = np.zeros((n_coalitions, 1000), dtype=float)

        for i in range(n_coalitions):
            coalition = coalitions[i]
            bool_masked_pos = self._transform_coalition_into_bool_mask(coalition, self.n_patches)
            with torch.no_grad():
                embeddings = self._embedding_layer(
                    **self._transformed_image, bool_masked_pos=bool_masked_pos
                )
                encodings = self._encoder(embeddings)
                norm_encodings = F.layer_norm(
                    encodings.last_hidden_state[:, 0],
                    self._norm_shape,
                    self._norm_weight,
                    self._norm_bias,
                    self._norm_eps,
                )
                logit_output = self._classifier(norm_encodings)
                probit_output = F.softmax(logit_output, dim=-1)

            probit_output_all[i] = probit_output.cpu().numpy()

        if self.class_id is not None:
            probit_output_all = probit_output_all[:, self.class_id]

        return probit_output_all

    @staticmethod
    def _transform_coalition_into_bool_mask(coalition: np.ndarray, n_patches: int) -> torch.Tensor:
        """Transforms a coalition of players (i.e. super-patches) into a boolean mask for the Vision
        Transformer model.

        The Vision Transformer model uses a 1D boolean mask to mask out patches that are not part of
        for the prediction. The underlying model operates on 12x12 (i.e. 144) patches. To reduce the
        number of players (i.e. patches), we group them together to form super-patches. The
        super-patches are then masked out jointly.

        Args:
            coalition: The coalition of players.
            n_patches: The number of patches used in the model.

        Returns:
            The boolean mask for the model in 1D with shape (144,).
        """
        bool_mask_2d = torch.ones((12, 12), dtype=torch.int)
        for player, is_present in enumerate(coalition):
            if is_present:
                if n_patches == 16:
                    x, y = (
                        MAPPING_PLAYER_MASK[16][player]["x"],
                        MAPPING_PLAYER_MASK[16][player]["y"],
                    )
                    bool_mask_2d[y : y + 3, x : x + 3] = False
                else:
                    x, y = MAPPING_PLAYER_MASK[9][player]["x"], MAPPING_PLAYER_MASK[9][player]["y"]
                    bool_mask_2d[y : y + 4, x : x + 4] = False
        bool_mask_1d = bool_mask_2d.flatten()
        return bool_mask_1d


# constants for the boolean mask generation for the Vision Transformer model
MAPPING_PLAYER_MASK = {
    16: {
        0: {"x": 0, "y": 0},
        1: {"x": 3, "y": 0},
        2: {"x": 6, "y": 0},
        3: {"x": 9, "y": 0},
        4: {"x": 0, "y": 3},
        5: {"x": 3, "y": 3},
        6: {"x": 6, "y": 3},
        7: {"x": 9, "y": 3},
        8: {"x": 0, "y": 6},
        9: {"x": 3, "y": 6},
        10: {"x": 6, "y": 6},
        11: {"x": 9, "y": 6},
        12: {"x": 0, "y": 9},
        13: {"x": 3, "y": 9},
        14: {"x": 6, "y": 9},
        15: {"x": 9, "y": 9},
    },
    9: {
        0: {"x": 0, "y": 0},
        1: {"x": 4, "y": 0},
        2: {"x": 8, "y": 0},
        3: {"x": 0, "y": 4},
        4: {"x": 4, "y": 4},
        5: {"x": 8, "y": 4},
        6: {"x": 0, "y": 8},
        7: {"x": 4, "y": 8},
        8: {"x": 8, "y": 8},
    },
}


NORM_WEIGHT = nn.Parameter(
    torch.Tensor(
        [
            0.7724,
            0.8678,
            0.9366,
            1.0039,
            1.0561,
            0.7385,
            1.0615,
            0.8865,
            0.8941,
            0.9424,
            0.8288,
            0.7922,
            1.0252,
            0.8710,
            0.9599,
            1.1121,
            0.8252,
            0.9334,
            0.8187,
            0.8979,
            0.8568,
            0.7679,
            0.9906,
            0.8347,
            0.9086,
            0.9159,
            0.8190,
            0.9097,
            0.0811,
            0.8481,
            0.8784,
            0.9600,
            0.8627,
            0.9671,
            0.8801,
            0.8387,
            0.9691,
            0.8524,
            1.0878,
            0.9591,
            1.0038,
            0.9874,
            0.7297,
            0.8289,
            0.8548,
            0.8613,
            0.9542,
            0.9356,
            0.8702,
            0.9186,
            1.1116,
            0.9478,
            0.7110,
            1.0284,
            0.8580,
            0.8528,
            0.9271,
            0.8582,
            0.9995,
            0.8160,
            0.9658,
            1.0855,
            0.9332,
            0.9192,
            1.0462,
            0.9247,
            0.8930,
            0.9687,
            1.0644,
            1.0803,
            0.8045,
            1.0416,
            1.0223,
            0.9203,
            0.8293,
            0.9262,
            0.9441,
            0.9536,
            0.9270,
            0.9764,
            0.7927,
            0.9274,
            0.0623,
            0.9641,
            0.8685,
            1.0562,
            0.9573,
            0.8327,
            0.8892,
            0.9693,
            0.3486,
            0.6806,
            0.8137,
            0.9927,
            0.8397,
            0.8657,
            1.0235,
            0.9955,
            0.9632,
            0.9005,
            0.9463,
            0.8557,
            0.9569,
            0.9334,
            1.0324,
            0.8787,
            1.0103,
            0.9511,
            0.6514,
            0.9038,
            0.8382,
            0.8588,
            0.6542,
            0.8213,
            0.9905,
            0.9300,
            0.7611,
            0.8986,
            0.9067,
            1.0513,
            0.9079,
            1.1033,
            0.8880,
            1.0064,
            0.8848,
            0.9266,
            0.9116,
            0.8185,
            0.8383,
            0.9301,
            0.8679,
            0.8738,
            0.9873,
            0.9393,
            0.9491,
            0.9263,
            0.9197,
            0.8703,
            0.9110,
            0.8636,
            1.1045,
            0.9407,
            1.0372,
            0.9653,
            0.9205,
            0.9503,
            1.0498,
            0.8616,
            0.8706,
            1.0054,
            0.9614,
            0.9837,
            0.9727,
            0.8497,
            0.8349,
            0.9096,
            1.1335,
            0.8498,
            1.1242,
            0.9407,
            0.9346,
            0.9110,
            0.9257,
            0.9563,
            0.8816,
            0.8089,
            0.8156,
            0.9790,
            0.8888,
            0.9688,
            0.8019,
            1.0013,
            1.1001,
            0.8499,
            0.9473,
            1.0339,
            0.9006,
            0.7905,
            0.8611,
            0.9615,
            1.0828,
            0.9820,
            0.9371,
            0.8533,
            0.9253,
            0.9303,
            0.9875,
            0.9780,
            0.8042,
            1.0080,
            0.7328,
            1.3287,
            1.0988,
            0.8406,
            0.8694,
            0.9579,
            1.0060,
            0.8466,
            0.9833,
            0.9054,
            1.1101,
            0.9187,
            0.8788,
            1.1473,
            0.7942,
            0.9605,
            0.7978,
            0.7996,
            0.8010,
            0.8592,
            0.8533,
            1.0013,
            0.8439,
            1.0221,
            0.8401,
            0.9757,
            1.0375,
            0.9133,
            0.8977,
            0.9861,
            0.8620,
            0.8918,
            0.9831,
            0.9109,
            0.9011,
            0.9795,
            0.9012,
            0.8763,
            0.9560,
            0.9355,
            0.9916,
            0.7737,
            0.8417,
            0.8781,
            0.9406,
            0.8081,
            0.9332,
            1.1916,
            0.9270,
            0.8294,
            0.9046,
            0.9548,
            0.9897,
            0.9223,
            0.9600,
            0.9564,
            0.9518,
            0.8534,
            0.7527,
            0.8416,
            0.9632,
            0.8388,
            1.0164,
            0.9667,
            1.0880,
            0.9616,
            0.9188,
            0.8545,
            0.8840,
            0.8956,
            0.9486,
            0.9026,
            0.8981,
            0.7480,
            0.9512,
            1.0520,
            0.9933,
            0.8824,
            0.9382,
            0.8843,
            0.9812,
            1.0271,
            0.9592,
            0.8372,
            0.9921,
            0.8706,
            0.7288,
            1.0285,
            0.7929,
            0.9721,
            0.9082,
            0.9727,
            0.8521,
            0.9294,
            0.7963,
            0.9024,
            1.1329,
            1.0402,
            1.0340,
            1.1268,
            0.8331,
            0.9055,
            0.8972,
            0.8544,
            0.8206,
            0.9281,
            0.9105,
            0.8359,
            0.8992,
            0.8534,
            0.9512,
            0.8994,
            0.8117,
            0.7603,
            0.7973,
            0.8292,
            0.8456,
            0.8781,
            0.9069,
            1.0577,
            1.0395,
            0.8889,
            0.9818,
            0.9843,
            1.0211,
            0.8104,
            0.8919,
            0.8306,
            0.8755,
            1.0086,
            0.9560,
            0.8305,
            0.2105,
            0.9681,
            0.8368,
            0.9242,
            1.0621,
            0.8657,
            0.8951,
            0.8638,
            0.8548,
            0.7956,
            0.9090,
            0.9435,
            1.0673,
            0.8951,
            1.0171,
            0.7729,
            0.9173,
            0.8867,
            0.9676,
            0.8437,
            0.8177,
            0.8727,
            0.8985,
            1.1087,
            0.8885,
            0.9941,
            0.8481,
            0.8944,
            0.7897,
            1.1287,
            1.0182,
            0.9361,
            0.9312,
            0.9137,
            1.0533,
            0.8580,
            0.9497,
            0.8149,
            1.0042,
            0.9867,
            0.8886,
            0.8384,
            0.8561,
            1.1610,
            1.0271,
            0.9036,
            0.8755,
            0.7572,
            0.7245,
            0.9052,
            1.0009,
            0.8929,
            0.8696,
            0.9699,
            0.9294,
            0.9880,
            1.0854,
            0.8865,
            0.9925,
            0.9193,
            0.6949,
            0.8343,
            0.9114,
            0.7377,
            0.9555,
            0.8486,
            0.9645,
            1.0320,
            1.1054,
            0.8194,
            0.8552,
            0.8497,
            0.8691,
            0.9128,
            0.9480,
            0.8429,
            0.9055,
            0.8477,
            0.9205,
            1.1300,
            1.0281,
            0.9729,
            0.9251,
            1.0230,
            0.9478,
            1.0413,
            0.8806,
            0.7511,
            0.8393,
            0.9495,
            0.8066,
            0.9952,
            0.9390,
            0.8451,
            1.0091,
            1.0213,
            1.0890,
            0.8349,
            0.9147,
            0.8191,
            0.8705,
            0.7890,
            0.9363,
            0.8504,
            0.8668,
            1.0688,
            0.9468,
            0.8352,
            0.9503,
            0.9715,
            0.8781,
            1.0943,
            1.1247,
            0.9537,
            0.7844,
            1.0475,
            0.8433,
            0.8837,
            0.9992,
            0.8478,
            1.0296,
            0.8582,
            0.7760,
            0.9323,
            0.9243,
            1.0490,
            0.9193,
            1.0968,
            0.9322,
            0.9289,
            0.8970,
            0.8519,
            0.9840,
            0.8700,
            0.7369,
            0.9200,
            1.0490,
            0.8925,
            0.9683,
            0.8302,
            0.9958,
            0.8893,
            0.9040,
            0.4897,
            0.8760,
            0.9123,
            0.8979,
            0.9581,
            0.9583,
            1.0101,
            0.8399,
            0.9703,
            0.9290,
            0.9713,
            0.8366,
            0.9345,
            0.7832,
            0.8965,
            0.8392,
            1.0287,
            0.9547,
            0.9832,
            0.8618,
            0.9595,
            0.8906,
            0.9399,
            0.9598,
            0.9023,
            0.9110,
            0.9155,
            0.8191,
            0.8432,
            0.8656,
            0.8365,
            0.9570,
            0.9998,
            0.8869,
            0.8008,
            0.9342,
            1.0209,
            1.0494,
            1.0277,
            0.9027,
            1.0171,
            0.9360,
            0.9374,
            0.8533,
            0.8405,
            1.1349,
            0.7831,
            0.8571,
            0.9954,
            1.0637,
            0.9654,
            0.8120,
            0.9684,
            0.9678,
            0.9516,
            0.7410,
            0.8906,
            0.8612,
            0.8992,
            0.9895,
            0.8555,
            0.8334,
            0.9515,
            0.9257,
            0.8949,
            0.8683,
            0.7349,
            0.9396,
            0.8399,
            0.9997,
            0.8872,
            1.0332,
            0.9636,
            0.8964,
            0.9037,
            0.9159,
            1.0100,
            0.7693,
            1.0079,
            1.2944,
            0.9369,
            0.9563,
            0.8877,
            0.7435,
            0.8477,
            0.8512,
            1.0185,
            0.8876,
            0.7201,
            0.8115,
            0.9732,
            1.0794,
            1.0951,
            0.7835,
            0.9369,
            0.8725,
            0.8147,
            0.8846,
            0.9998,
            0.9889,
            0.8987,
            0.9635,
            0.9289,
            1.0650,
            0.9126,
            0.9303,
            1.0126,
            0.8792,
            0.8913,
            0.8935,
            0.8865,
            1.0142,
            0.2996,
            0.9205,
            0.8239,
            1.0939,
            0.8672,
            0.7174,
            0.9033,
            0.9012,
            0.8639,
            1.1006,
            1.4171,
            0.7527,
            1.1798,
            0.9635,
            1.0166,
            0.9037,
            0.9642,
            0.8659,
            1.0022,
            1.0488,
            0.8736,
            0.9192,
            0.8422,
            0.9307,
            0.7862,
            0.7634,
            0.9518,
            0.9802,
            0.9648,
            0.8098,
            0.7833,
            0.7797,
            0.8842,
            0.8924,
            0.9377,
            0.9760,
            0.8014,
            0.9804,
            0.9231,
            0.8991,
            0.7893,
            0.8682,
            0.9154,
            1.4090,
            0.9018,
            0.9509,
            0.8707,
            0.8127,
            0.9029,
            1.0098,
            0.8448,
            0.9611,
            0.9249,
            1.0108,
            0.8216,
            0.9399,
            0.9445,
            0.8045,
            1.5534,
            1.0799,
            0.8302,
            0.9203,
            0.9762,
            0.9494,
            0.9449,
            1.0336,
            0.9672,
            0.9501,
            0.8223,
            1.0093,
            0.9727,
            0.9178,
            0.8226,
            0.8792,
            0.9698,
            0.9362,
            0.9734,
            0.9786,
            1.0469,
            0.8709,
            0.7950,
            0.8859,
            0.8224,
            0.9697,
            0.9445,
            0.9815,
            0.8978,
            0.8381,
            0.9834,
            0.8826,
            0.8739,
            0.9176,
            0.8350,
            0.8208,
            0.7715,
            1.0389,
            0.8091,
            0.8650,
            1.0885,
            0.8841,
            0.7818,
            0.9085,
            0.9270,
            0.8804,
            0.9452,
            0.9685,
            0.8172,
            0.8619,
            0.9601,
            1.0949,
            0.7882,
            0.8938,
            0.8152,
            0.8450,
            0.8052,
            0.9305,
            1.0294,
            0.9271,
            1.0985,
            0.8878,
            0.8054,
            1.1544,
            0.8261,
            0.9380,
            1.1291,
            0.8410,
            0.8618,
            0.7883,
            0.8794,
            0.9373,
            1.0322,
            1.0609,
            0.9702,
            1.0714,
            1.0356,
            0.8360,
            0.8451,
            0.8794,
            0.8118,
            0.8697,
            0.8049,
            0.8704,
            0.8380,
            0.8226,
            0.9955,
            0.9289,
            0.9157,
            0.9386,
            0.9267,
            0.9261,
            0.7927,
            0.7923,
            0.9631,
            0.9301,
            1.1386,
            0.7755,
            0.9143,
            0.8176,
            0.9678,
            1.0312,
            0.8141,
            0.8435,
            0.9756,
            0.9107,
            0.8000,
            0.8897,
            0.9853,
            0.7918,
            0.9263,
            1.0086,
            0.8121,
            0.9490,
            0.8851,
            0.7554,
            1.0374,
            0.9236,
            0.9658,
            0.8252,
            1.0178,
            1.0434,
            0.9084,
            0.8683,
            0.8403,
            0.9014,
            0.9790,
            1.0228,
            0.9348,
            0.9140,
            0.7513,
            0.9935,
            0.8690,
        ]
    )
)

NORM_BIAS = nn.Parameter(
    torch.Tensor(
        [
            -1.2846e-02,
            -3.4329e-02,
            -1.5974e-02,
            -4.2982e-02,
            -3.0238e-02,
            -9.0798e-02,
            1.4726e-03,
            -4.4224e-02,
            1.4199e-02,
            -4.4211e-02,
            -6.5910e-02,
            -2.9228e-02,
            -5.2106e-02,
            -2.8973e-02,
            -2.1046e-02,
            -3.5557e-02,
            -4.9552e-02,
            -5.8253e-02,
            -6.7512e-02,
            3.6097e-02,
            -6.5807e-02,
            -4.5931e-02,
            -4.3622e-02,
            -2.9615e-02,
            -4.9325e-03,
            -5.8695e-02,
            -7.9064e-02,
            -4.0975e-02,
            -1.8755e-01,
            -4.1807e-02,
            -4.9466e-02,
            -4.0243e-02,
            -3.6462e-02,
            -3.7576e-02,
            1.2360e-02,
            -4.9116e-02,
            -2.0148e-02,
            -1.2291e-02,
            -6.4720e-04,
            -1.5631e-02,
            3.5349e-02,
            -5.3380e-02,
            -5.2839e-02,
            -3.5356e-02,
            -4.9360e-02,
            -3.7034e-02,
            -2.4196e-02,
            -2.2688e-02,
            -5.7503e-02,
            -1.7312e-02,
            -2.1079e-02,
            -3.1677e-02,
            4.8169e-02,
            -3.1830e-02,
            -3.8588e-02,
            4.3348e-03,
            5.0198e-03,
            -1.4342e-02,
            1.3782e-02,
            -8.3937e-02,
            -4.0401e-02,
            -4.4845e-04,
            -4.3068e-02,
            -4.2351e-02,
            -6.0829e-02,
            -5.2694e-02,
            6.8000e-03,
            -4.9412e-02,
            -2.8671e-02,
            -3.9205e-02,
            -4.1686e-02,
            2.6699e-02,
            -6.9093e-02,
            -2.8231e-02,
            -5.0052e-02,
            -2.0931e-02,
            -4.4256e-02,
            1.7926e-02,
            -3.9110e-02,
            -3.9005e-02,
            -2.9385e-02,
            5.8100e-03,
            -1.7442e-01,
            -4.1924e-02,
            -1.9087e-02,
            -8.0975e-02,
            -4.5490e-02,
            3.1678e-02,
            -3.9018e-02,
            -3.1937e-02,
            2.1715e-01,
            1.1593e-01,
            -3.7255e-02,
            -3.6731e-02,
            1.4943e-01,
            -3.9838e-02,
            -3.3936e-03,
            -7.4650e-02,
            -3.2300e-02,
            -6.2477e-02,
            -4.3892e-02,
            -2.1577e-02,
            -1.3362e-02,
            -1.5035e-03,
            -1.1464e-02,
            1.1086e-02,
            4.0921e-02,
            -2.9174e-02,
            -7.9901e-02,
            -5.0198e-02,
            -4.7126e-02,
            -2.6920e-02,
            5.5239e-02,
            -5.2750e-04,
            1.1021e-02,
            -1.0059e-02,
            -1.4993e-02,
            -5.1686e-02,
            -7.9651e-02,
            -5.9970e-02,
            -8.1743e-02,
            -1.6672e-02,
            -4.1509e-02,
            -3.6090e-02,
            -1.1362e-02,
            -1.4957e-04,
            -1.6178e-02,
            -3.7019e-02,
            -2.6179e-02,
            -2.4196e-02,
            -5.9354e-02,
            -5.1076e-02,
            -5.4872e-02,
            3.6121e-02,
            -5.8862e-02,
            -4.5470e-02,
            -1.5193e-02,
            -2.7477e-02,
            1.6383e-02,
            7.3675e-03,
            -7.9882e-02,
            -4.8497e-02,
            -4.9284e-02,
            -8.5378e-03,
            -3.5176e-02,
            -1.6944e-02,
            -5.6284e-02,
            4.5611e-03,
            -7.9809e-02,
            -2.0827e-02,
            2.1044e-02,
            2.5901e-02,
            -1.2585e-02,
            -1.7256e-02,
            -7.6534e-03,
            1.2117e-03,
            4.5330e-02,
            -1.3189e-02,
            1.1903e-02,
            -9.4587e-03,
            2.6063e-02,
            -4.1669e-02,
            3.6421e-02,
            -4.1968e-02,
            -2.0173e-02,
            2.9693e-02,
            -2.2086e-02,
            1.6400e-02,
            -3.0243e-02,
            -7.2582e-02,
            -3.7255e-02,
            3.1800e-02,
            2.5285e-02,
            -5.9659e-02,
            1.4987e-03,
            -3.3993e-02,
            -3.1998e-02,
            -2.9844e-02,
            -1.0384e-02,
            -5.5173e-02,
            -2.4426e-02,
            -6.0636e-02,
            -4.3924e-02,
            -2.1287e-03,
            -7.8432e-02,
            -4.1421e-03,
            -7.7996e-03,
            -1.1429e-02,
            -4.6941e-02,
            -8.7613e-02,
            -6.4707e-02,
            -3.9325e-01,
            -2.6667e-02,
            -4.2468e-02,
            2.8707e-02,
            -5.3469e-02,
            -4.2843e-03,
            -2.5787e-02,
            -5.8231e-03,
            -1.3993e-03,
            -2.8768e-02,
            -3.5016e-02,
            -7.2760e-02,
            -4.7598e-02,
            -2.6079e-02,
            -8.8605e-03,
            -3.6690e-02,
            -1.0348e-01,
            -3.2959e-02,
            -9.8152e-03,
            -5.1905e-02,
            -4.4468e-02,
            -6.1405e-02,
            2.7618e-03,
            8.6430e-03,
            -1.5917e-02,
            -4.6791e-02,
            -7.9614e-02,
            -4.6663e-02,
            -4.8553e-02,
            -2.2205e-02,
            -4.9833e-02,
            -1.6760e-02,
            3.0068e-02,
            -2.6810e-02,
            -1.6550e-02,
            -3.1726e-02,
            2.2589e-02,
            1.9759e-02,
            1.9460e-02,
            -1.5823e-02,
            -4.5633e-02,
            7.0996e-03,
            -5.3303e-02,
            -1.4067e-02,
            3.6364e-02,
            -2.7200e-02,
            4.4156e-01,
            -1.8705e-02,
            -2.9289e-02,
            -3.6419e-02,
            -5.7678e-02,
            -3.9713e-02,
            -1.3220e-02,
            -4.5488e-02,
            -6.4137e-02,
            -2.7480e-02,
            3.1830e-02,
            2.8179e-03,
            -1.9385e-02,
            2.5883e-03,
            -5.4950e-02,
            -1.0737e-01,
            3.7381e-03,
            -3.3766e-02,
            -5.0923e-02,
            1.7538e-02,
            -3.0714e-02,
            -2.9862e-02,
            -6.9823e-03,
            -5.1923e-02,
            -8.2818e-02,
            -3.4587e-02,
            1.8858e-03,
            -1.0735e-02,
            -2.2856e-02,
            -1.8102e-02,
            -1.7566e-02,
            2.5976e-02,
            -2.7338e-02,
            -2.2985e-02,
            -1.1991e-02,
            -4.8576e-02,
            1.9924e-02,
            -3.1608e-02,
            -1.4714e-02,
            -5.5489e-02,
            -5.2503e-02,
            -1.6846e-02,
            -6.7046e-02,
            2.2597e-02,
            -6.9730e-02,
            -3.8120e-02,
            -1.2192e-02,
            9.5660e-03,
            -9.3410e-02,
            -5.7038e-02,
            -4.5223e-02,
            -1.9107e-03,
            -4.5653e-02,
            -3.1560e-02,
            -7.0479e-03,
            1.6093e-02,
            -1.4466e-02,
            -1.7401e-02,
            -5.5894e-02,
            -4.9992e-02,
            -2.7360e-02,
            -5.4941e-02,
            -6.1053e-02,
            -3.6248e-02,
            -8.4022e-03,
            -6.2174e-02,
            -9.2947e-03,
            -6.1944e-02,
            -6.5625e-02,
            -2.8564e-02,
            -2.2109e-02,
            1.0073e-03,
            4.1944e-02,
            -1.1042e-02,
            -4.8359e-02,
            -4.0167e-02,
            -5.2894e-04,
            -6.3634e-02,
            -2.7082e-02,
            -4.0973e-02,
            -1.2798e-02,
            -1.3506e-02,
            -3.3824e-02,
            -3.8021e-02,
            -6.6481e-02,
            2.2004e-01,
            -4.1205e-02,
            -4.1045e-02,
            -6.7147e-03,
            -9.7474e-03,
            -6.6682e-02,
            -5.0532e-02,
            1.7950e-02,
            -3.2157e-02,
            -2.8190e-02,
            -7.8217e-02,
            -2.7831e-02,
            -5.7644e-02,
            6.6083e-03,
            -6.3298e-02,
            -4.6771e-02,
            -4.0984e-02,
            -3.2495e-02,
            -2.1753e-02,
            8.4077e-03,
            -6.1491e-03,
            -7.5623e-02,
            -2.5362e-02,
            -6.0820e-02,
            -4.6130e-02,
            -7.2358e-02,
            -3.2118e-04,
            -2.4742e-02,
            -3.7831e-02,
            -6.0944e-02,
            -3.8093e-02,
            -5.5954e-02,
            -4.1207e-02,
            -4.2415e-02,
            -6.6750e-02,
            -3.8377e-02,
            -2.6787e-02,
            -3.0111e-02,
            -5.5692e-02,
            1.9226e-03,
            -3.5744e-02,
            -1.7536e-02,
            2.1400e-03,
            3.4993e-02,
            -5.4584e-02,
            -5.5105e-02,
            -4.9076e-02,
            -7.6570e-02,
            2.1697e-02,
            2.7359e-02,
            -8.0934e-03,
            -4.0660e-02,
            -5.1813e-02,
            -4.1121e-02,
            -1.8887e-02,
            5.6397e-02,
            -5.8658e-03,
            -9.6132e-03,
            -8.1223e-02,
            -9.2672e-03,
            -3.3466e-02,
            -2.2833e-03,
            -5.1733e-02,
            -6.2478e-02,
            -7.1157e-02,
            -1.5965e-02,
            -3.4498e-02,
            -4.0631e-02,
            -7.1113e-02,
            -2.5857e-02,
            -1.0533e-02,
            -7.1634e-02,
            -7.4607e-02,
            -4.5122e-03,
            -2.4341e-02,
            -2.3021e-02,
            -2.1421e-02,
            -3.7153e-02,
            2.5136e-02,
            -5.4707e-02,
            -2.2787e-02,
            -4.6770e-03,
            -1.8227e-03,
            2.9149e-03,
            -7.3360e-02,
            -2.5106e-03,
            -2.9604e-02,
            -4.8027e-03,
            -4.0431e-02,
            -2.9706e-02,
            -7.9295e-02,
            -3.4943e-02,
            1.4558e-03,
            1.0653e-02,
            -2.7563e-02,
            1.6224e-02,
            1.7811e-02,
            -5.1659e-02,
            4.7950e-03,
            -8.3416e-02,
            -2.2950e-02,
            -2.1858e-02,
            7.0907e-03,
            -7.4321e-02,
            -4.6718e-02,
            1.1171e-02,
            3.7422e-02,
            -7.0223e-02,
            1.5610e-02,
            -4.9681e-02,
            -3.8607e-02,
            -4.2889e-02,
            1.6026e-04,
            -6.8360e-02,
            2.7597e-03,
            -1.5055e-02,
            -7.8066e-02,
            -1.1554e-02,
            -1.0518e-01,
            -2.6719e-02,
            -6.8114e-02,
            -4.8772e-02,
            -4.1149e-02,
            -5.6604e-03,
            -5.5007e-03,
            -1.4997e-03,
            2.1207e-02,
            2.5436e-02,
            -1.6502e-02,
            -4.3622e-02,
            -3.1551e-02,
            -4.8746e-02,
            -5.4541e-02,
            -1.9924e-02,
            3.5569e-03,
            -2.2993e-02,
            -2.9776e-02,
            1.5344e-03,
            -5.2402e-02,
            -2.8452e-02,
            -9.5660e-03,
            -7.8062e-02,
            -4.9867e-02,
            1.2912e-01,
            -3.9901e-02,
            2.0674e-02,
            -8.6055e-03,
            -4.5842e-02,
            -2.5712e-02,
            -5.4000e-03,
            -2.4815e-02,
            -1.6062e-02,
            1.2359e-02,
            -2.3535e-02,
            -1.0057e-02,
            2.3549e-03,
            -1.6457e-02,
            -5.9837e-02,
            -1.3972e-02,
            -2.7198e-02,
            -3.8551e-02,
            -4.3705e-02,
            -9.3202e-03,
            -1.0222e-01,
            -4.5649e-02,
            2.2186e-02,
            -4.2121e-03,
            -6.3025e-02,
            1.4204e-02,
            1.8401e-02,
            -4.6681e-02,
            -1.3743e-02,
            -7.1678e-03,
            -4.3146e-02,
            3.8278e-02,
            -3.4152e-02,
            -4.1673e-02,
            2.0328e-02,
            -4.7003e-02,
            -3.9247e-03,
            -4.4714e-02,
            -4.8451e-02,
            -4.5147e-02,
            -4.5608e-02,
            4.6197e-03,
            -3.3829e-02,
            -5.4287e-02,
            2.1881e-02,
            -4.5251e-02,
            -5.4296e-02,
            -3.9472e-02,
            -7.9726e-02,
            -4.5267e-02,
            -1.6026e-02,
            -4.3071e-02,
            -9.7591e-03,
            -5.8285e-02,
            -2.3282e-02,
            -8.6472e-03,
            -7.6034e-03,
            3.9871e-03,
            -7.9120e-02,
            2.2551e-02,
            -8.7009e-03,
            -3.5223e-02,
            -1.1489e-02,
            -1.5564e-02,
            -6.5891e-02,
            -3.5131e-02,
            2.6181e-02,
            7.9049e-03,
            -1.9506e-02,
            -4.3038e-02,
            -1.4197e-02,
            -4.1955e-02,
            -6.6071e-02,
            -1.4759e-02,
            -4.1931e-02,
            -2.7656e-02,
            2.1169e-02,
            1.6531e-02,
            -3.8823e-02,
            3.6122e-01,
            -1.3912e-02,
            -4.6448e-02,
            -5.6456e-02,
            1.8823e-02,
            4.2217e-02,
            7.7383e-03,
            -1.6559e-02,
            -2.1881e-02,
            -7.0389e-02,
            -9.1009e-03,
            -9.3070e-02,
            -2.7914e-02,
            -8.7950e-03,
            -7.1352e-02,
            -1.7040e-02,
            3.1406e-04,
            -4.5131e-02,
            -4.6582e-02,
            -5.9288e-02,
            1.5186e-02,
            -6.4614e-02,
            -4.2326e-02,
            6.4071e-03,
            4.3422e-02,
            -4.9171e-02,
            -6.0609e-02,
            -6.8295e-02,
            -2.3742e-02,
            -2.1920e-02,
            2.0444e-02,
            -2.7487e-02,
            -3.7172e-02,
            2.0728e-01,
            3.0919e-02,
            -9.3169e-02,
            -6.4641e-02,
            -1.0803e-02,
            2.9469e-02,
            -4.8751e-02,
            -2.0309e-02,
            -1.6740e-02,
            -3.1540e-02,
            2.0689e-01,
            4.6825e-02,
            -1.7657e-02,
            -5.8028e-02,
            -2.0241e-02,
            -4.1637e-02,
            -6.1978e-02,
            -6.5709e-02,
            -3.0832e-02,
            1.1827e-02,
            -1.9401e-02,
            -4.4029e-02,
            -6.7469e-02,
            -9.4239e-03,
            -3.7024e-02,
            -7.7362e-02,
            -1.2563e-02,
            -2.8132e-03,
            -3.6103e-02,
            -2.5216e-02,
            -7.3705e-02,
            -1.7333e-02,
            -2.9617e-02,
            -2.9742e-02,
            -1.4596e-02,
            -1.3917e-02,
            -6.2427e-02,
            4.0526e-02,
            -5.9457e-02,
            -4.7396e-02,
            -9.0681e-03,
            -4.5602e-02,
            -2.8426e-03,
            -1.4107e-02,
            2.5676e-02,
            -2.9223e-02,
            4.7225e-03,
            9.1192e-04,
            -2.9369e-02,
            -1.9747e-02,
            -1.2733e-02,
            -1.6912e-02,
            -5.2998e-02,
            -3.7847e-02,
            -1.9489e-02,
            -4.9093e-02,
            -4.3426e-02,
            -3.1547e-02,
            2.0800e-01,
            -3.2695e-02,
            -2.1654e-02,
            -1.1146e-02,
            -6.5462e-02,
            -2.3371e-02,
            -2.6509e-02,
            3.2668e-02,
            1.0064e-02,
            -2.0494e-02,
            -5.5164e-02,
            -6.7452e-02,
            -2.8337e-02,
            3.5766e-02,
            -2.9156e-02,
            -2.3041e-02,
            -1.4671e-02,
            -3.5276e-02,
            -5.1762e-02,
            -1.9228e-02,
            -6.2443e-02,
            -5.2437e-02,
            -3.1743e-02,
            -1.2163e-02,
            -6.8294e-03,
            -3.7384e-02,
            -4.2817e-02,
            -1.8869e-03,
            2.0390e-02,
            -2.9747e-02,
            -5.1032e-02,
            -1.9200e-02,
            -7.5637e-02,
            -6.8009e-02,
            -8.0360e-02,
            -7.1139e-02,
            -2.0064e-02,
            1.4154e-03,
            6.2020e-03,
            -7.3787e-02,
            3.3159e-02,
            -2.1919e-02,
            2.4633e-02,
            1.2704e-03,
            1.5340e-04,
            1.5592e-02,
            -5.5488e-02,
            -2.9665e-02,
            -1.1985e-02,
            6.7323e-04,
            -5.4133e-02,
            3.4998e-02,
            -4.5131e-02,
            -4.6766e-03,
            5.1330e-03,
            -3.4061e-02,
            -9.3064e-03,
            -4.7291e-02,
            -7.0864e-02,
            -4.1527e-02,
            1.1926e-03,
            -4.6581e-02,
            -2.8749e-02,
            -7.9662e-02,
            -2.7212e-02,
            -1.7270e-02,
            -7.8053e-02,
            -8.3372e-02,
            -5.9839e-02,
            -5.4487e-02,
            2.1755e-02,
            -1.7783e-02,
            -6.7328e-02,
            2.5955e-03,
            -3.2142e-02,
            -1.7771e-02,
            -5.3142e-02,
            -5.0849e-02,
            2.3232e-02,
            -4.6922e-02,
            -3.0984e-02,
            -2.3301e-02,
            -3.1234e-02,
            -8.6854e-02,
            -2.6192e-02,
            -5.3706e-02,
            -3.3049e-02,
            2.4952e-02,
            -2.9323e-02,
            5.2988e-02,
            3.4971e-02,
            -7.4179e-02,
            3.4402e-02,
            -1.6393e-02,
            -2.7252e-02,
            -4.8912e-04,
            -7.8547e-03,
            -6.1618e-02,
            -4.3479e-02,
            1.7533e-03,
            -6.6096e-02,
            -1.7410e-02,
            -3.0186e-03,
            -3.9912e-02,
            5.7701e-03,
            -6.3653e-02,
            -5.3616e-02,
            -2.8609e-02,
            -1.5203e-02,
            7.0197e-03,
            -1.0869e-01,
            -3.6286e-02,
            -2.0196e-02,
            -6.4744e-02,
            -6.3095e-02,
            -5.9029e-03,
            -4.7942e-02,
            -6.8610e-03,
            -2.0051e-02,
            4.0774e-02,
            -6.2706e-02,
            -5.0411e-03,
            -2.0793e-02,
            -3.4539e-02,
            -4.4209e-02,
            -3.2647e-02,
            -3.4269e-02,
            3.6976e-02,
            -2.3941e-02,
            -4.4631e-02,
            1.6395e-02,
            -8.8686e-02,
            5.7581e-02,
        ]
    )
)
